using System;
using System.IO;
using System.Xml;
using System.Diagnostics;

using Team_Project.Exceptions;

namespace Team_Project.PersistencyManagers.Storages
{
	/// <summary>
	/// Storage utilizzato per la persistenza su file system locale.
	/// </summary>
	public class LocalFileSystemStorage: MarshalByRefObject, IStorage
	{
		/// <summary>
		/// Percorso base dalla quale  replicato nel file system la strutture
		/// delle locations
		/// </summary>
		protected string Base;

		/// <summary>
		/// Costruisce uno storage configurandolo con i dati letti dal nodo
		/// in ingresso
		/// </summary>
		/// <param name="cfg">Nodo xml contenente i dati necessari per la
		/// configurazione del nodo</param>
		public LocalFileSystemStorage(XmlNode cfg)
		{
			XmlNode bp = cfg.SelectSingleNode("BasePath");
			string baseDir = bp.Attributes["value"].Value;
			if(!baseDir.EndsWith(@"\"))
				baseDir += @"\";
			Base = baseDir;
			if(!Directory.Exists(Base))
				Directory.CreateDirectory(Base);
		}

		/// <summary>
		/// Costruisce il percorso completo del file che contiene l'elemento
		/// </summary>
		/// <param name="Location">Location dell'elemento</param>
		/// <param name="Name">Nome dell'elemento</param>
		/// <returns>Il percorso completo del file che contiene i dati
		/// dell'elemento</returns>
		protected string BuildFullName(string Location,string Name)
		{
			string fullName = Base;
			if(Location != string.Empty && !Location.EndsWith(@"\"))
				return Base + Location + @"\" + Name;
			else return Base + Location + Name;
		}

		#region IStorage Members
		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.OpenRead">
		/// IStorage.OpenRead</see>
		/// </summary>
		/// <param name="Location"></param>
		/// <param name="Name"></param>
		/// <returns></returns>
		public System.IO.Stream OpenRead(string Location, string Name)
		{
			if(!Directory.Exists(Base + Location))
				throw new LocationNotExistsException(Location.Substring(0,Location.Length));
			string fullName = BuildFullName(Location,Name);
			if(!File.Exists(fullName))
				throw new ElementNotExistsException(Location + " " + Name);
			return File.OpenRead(fullName);
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.LocationsList">
		/// IStorage.LocationsList</see>
		/// </summary>
		/// <param name="baseL"></param>
		/// <returns></returns>
		public string[] LocationsList(string baseL)
		{
			string fullName = BuildFullName(baseL,"");
			if(!Directory.Exists(fullName))
				throw new LocationNotExistsException(baseL);
			string[] dirPaths = Directory.GetDirectories(fullName);
			string[] dirNames = new string[dirPaths.Length];
			for(int i = 0; i < dirPaths.Length; i++)
			{
				DirectoryInfo di = new DirectoryInfo(dirPaths[i]);
				dirNames[i] = di.Name;
			}
			return dirNames;
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.OpenWrite">
		/// IStorage.OpenWrite</see>
		/// </summary>
		/// <param name="Location"></param>
		/// <param name="Name"></param>
		/// <returns></returns>
		public System.IO.Stream OpenWrite(string Location, string Name)
		{
			if(!Directory.Exists(Base + Location))
				throw new LocationNotExistsException(Location.Substring(0,Location.Length-1));
			
			string fullName = BuildFullName(Location,Name);
			if(File.Exists(fullName))
			{
				File.Delete(fullName);
			}
			return File.Create(fullName);
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.LocationExists">
		/// IStorage.LocationExists</see>
		/// </summary>
		/// <param name="Location"></param>
		/// <returns></returns>
		public bool LocationExists(string Location)
		{
			return Directory.Exists(Base + Location);
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.Exists">
		/// IStorage.Exists</see>
		/// </summary>
		/// <param name="Location"></param>
		/// <param name="Name"></param>
		/// <returns></returns>
		public bool Exists(string Location, string Name)
		{
			return File.Exists(BuildFullName(Location,Name));
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.CreateLocation">
		/// IStorage.CreateLocation</see>
		/// </summary>
		/// <param name="Location"></param>
		public void CreateLocation(string Location)
		{
			Trace.WriteLine("Requested creation of location " + Location);
			if(!LocationExists(Location))
			{
				Directory.CreateDirectory(Base + Location);
				Trace.WriteLine("Location " + Location + " created");
			}
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.DestroyLocation">
		/// IStorage.DestroyLocation</see>
		/// </summary>
		/// <param name="Location"></param>
		/// <param name="checkEmpty"></param>
		public void DestroyLocation(string Location, bool checkEmpty)
		{
			DirectoryInfo di = new DirectoryInfo(Base + Location);
			if(checkEmpty)
			{
				if(di.GetFiles().Length != 0 || di.GetDirectories().Length != 0)
					throw new LocationNotEmptyException(Location,"Cannot delete location because it isn't empty");
			}
			di.Delete(true);
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.Delete">
		/// IStorage.Delete</see>
		/// </summary>
		/// <param name="Location"></param>
		/// <param name="Name"></param>
		public void Delete(string Location, string Name)
		{
			File.Delete(BuildFullName(Location,Name));
		}

		/// <summary>
		/// Vedi <see cref="Team_Project.PersistencyManagers.Storages.IStorage.ElementsIn">
		/// IStorage.ElementsIn</see>
		/// </summary>
		/// <param name="Location"></param>
		/// <returns></returns>
		public string[] ElementsIn(string Location)
		{
			string[] filePaths = Directory.GetFiles(Base + Location);
			string[] fileNames = new string[filePaths.Length];
			for(int i = 0; i < filePaths.Length; i++)
			{
				FileInfo fi = new FileInfo(filePaths[i]);
				fileNames[i] = fi.Name;
			}
			return fileNames;
		}

		#endregion
	}
}
